' SAP Navigation Back Script
' Author: Dario Pascoal
'
' Description: This script handles navigation back to the main SAP screen after
' documents or transactions have been completed or closed. This is essential
' in SAP automation workflows where we need to return to a clean state before
' starting new operations.
'
' The script performs these main functions:
' 1. Connects to an existing SAP GUI session (doesn't create a new one)
' 2. Navigates back through the SAP menu hierarchy using the back button
' 3. Performs multiple back operations to handle various transaction depths
' 4. Logs all operations for debugging and audit purposes
'
' This script is typically called at the end of other SAP automation scripts
' to ensure the SAP session is left in a clean, predictable state for the
' next operation. It's designed to be robust and handle different SAP
' transaction contexts.
'
' Prerequisites:
' - SAP GUI must be running with an active session
' - SAP GUI scripting must be enabled
' - Optional: Log file path can be passed as command line argument
'
' Usage: cscript sap-navigate-back.vbs [optional_log_file_path]

On Error Resume Next

' =============================================================================
' SECTION 1: LOG FILE INITIALIZATION AND SETUP
' =============================================================================
' This section handles the optional logging functionality. If a log file path
' is provided as a command line argument, we set up file-based logging to
' track what the script is doing. This is very helpful for debugging when
' SAP automation scripts don't work as expected.

' Declare log file object and path
Dim logFile, logFilePath

' Initialize log file if path is provided
' Check if the user provided a log file path when running this script
If WScript.Arguments.Count > 0 Then
    ' Get the log file path from the first command line argument
    logFilePath = WScript.Arguments.Item(0)
    
    ' Try to open the log file in append mode (8 means append, True means create if doesn't exist)
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Check if we successfully opened the log file
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script start with clear separators for easy reading
        LogToFile "========================================="
        LogToFile "Navigate Back Script Started at " & Now
        LogToFile "========================================="
    End If
End If

' =============================================================================
' SECTION 2: LOGGING HELPER FUNCTIONS
' =============================================================================
' These functions provide a consistent way to log messages both to the console
' (so you can see what's happening when running the script) and to a file
' (so you can review what happened later for debugging).

' Function to log message to file only
' This writes a message to the log file with timestamp and script name
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Format: timestamp - [script name] - message
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Function to log message to both console and file
' This shows the message on screen AND writes it to the log file
Sub LogMessage(message)
    WScript.Echo message  ' Show on screen
    LogToFile message     ' Write to log file
End Sub

' =============================================================================
' SECTION 3: SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Before we can automate SAP, we need to disable the security warnings that
' SAP GUI shows when scripts try to connect. These registry settings tell
' SAP GUI to trust our automation scripts without showing popup warnings.

' Set scripting mode to true
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting warnings in registry
' These settings prevent SAP GUI from showing security popups that would
' interrupt our automation. We're essentially telling SAP GUI "trust this script"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' =============================================================================
' SECTION 4: SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' This section connects to an existing SAP GUI session. Unlike other scripts
' that might start SAP GUI, this script assumes SAP is already running and
' just connects to the existing session to perform the navigation back.

' Connect to existing SAP session
' First, try to get the main SAP GUI automation object
If Not IsObject(application) Then
   Set SapGuiAuto  = GetObject("SAPGUI")
   If Err.Number <> 0 Then
       LogMessage "ERROR: Cannot connect to SAP - " & Err.Description
       WScript.Quit 1
   End If
   ' Get the scripting engine from SAP GUI
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Get the connection (usually there's only one active connection)
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If

' Get the session (usually there's only one active session per connection)
If Not IsObject(session) Then
   Set session = connection.Children(0)
End If

' =============================================================================
' SECTION 5: SAP NAVIGATION BACK OPERATION
' =============================================================================
' This is the main business logic of the script. We navigate back through the
' SAP transaction hierarchy to return to the main screen. We do this multiple
' times because different SAP transactions have different depths, and we want
' to make sure we get back to a clean starting point.

' Navigate back multiple times to ensure we get to the main screen
' The back button in SAP GUI toolbar is btn[3] - this is the standard "Back" button
' We do this 3 times to handle different transaction depths:
' - Some transactions might be 1 level deep (need 1 back)
' - Others might be 2-3 levels deep (need multiple backs)
' - Better to do too many backs than too few
On Error Resume Next
For i = 1 To 3
    ' Press the back button (btn[3] is the back button in SAP GUI toolbar)
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    ' Wait 300 milliseconds between each back operation
    ' This gives SAP time to process the navigation before the next back
    WScript.Sleep 300
Next

LogMessage "SUCCESS: SAP navigation completed"

' =============================================================================
' SECTION 6: CLEANUP AND LOG FILE CLOSURE
' =============================================================================
' This section handles proper cleanup of the script, especially closing the
' log file if we were writing to one. Proper cleanup prevents file locks and
' ensures all log messages are actually written to disk.

' Function to properly close the log file
' This ensures all logged information is saved and the file is properly closed
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write final log entries to mark the end of this script's execution
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        ' Close the file and release the file handle
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

' Call the cleanup function to properly end the script
CloseLogFile()